// app.js - MikroTik Hotspot FULL (Single Router)

const express = require("express");
const cors = require("cors"); // <--- MUDANÇA 1: Importa o middleware CORS
const { RouterOSAPI } = require("node-routeros");

const app = express();
app.use(cors()); // <--- MUDANÇA 2: Adiciona o middleware CORS para permitir requisições de outras origens
app.use(express.json());

// CONFIG
const host = process.env.MIKROTIK_HOST || "201.62.94.182";
const user = process.env.MIKROTIK_USER || "admin";
const password = process.env.MIKROTIK_PASS || "amtu@123";
const PORT = process.env.PORT || 3000;
const API_KEY = process.env.API_KEY || "bbry123456";

// MIDDLEWARE
function requireApiKey(req, res, next) {
  const key = req.headers["x-api-key"];
  if (!key || key !== API_KEY) return res.status(401).json({ error: "Unauthorized" });
  next();
}

function mk() {
  return new RouterOSAPI({
    host,
    user,
    password,
    port: 8728,
    timeout: 7000
  });
}

async function safeConnect(api) {
  try {
    await api.connect();
    return true;
  } catch (e) {
    throw new Error("Failed to connect to MikroTik: " + e.message);
  }
}

// HEALTH
app.get("/api/health", (req, res) => res.json({ ok: true, host }));

// SYSTEM INFO
app.get("/api/system/info", requireApiKey, async (req, res) => {
  const api = mk();
  try {
    await safeConnect(api);
    const [identity, resource] = await Promise.all([
      api.write("/system/identity/print"),
      api.write("/system/resource/print")
    ]);
    api.close();
    res.json({ identity, resource });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// BACKUP SAVE
app.post("/api/system/backup/save", requireApiKey, async (req, res) => {
  const { name } = req.body;
  if (!name) return res.status(400).json({ error: "name required" });
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/system/backup/save", [`=name=${name}`]);
    api.close();
    res.json({ status: "backup_saved", name });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// BACKUP LOAD
app.post("/api/system/backup/load", requireApiKey, async (req, res) => {
  const { name } = req.body;
  if (!name) return res.status(400).json({ error: "name required" });
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/system/backup/load", [`=name=${name}`]);
    api.close();
    res.json({ status: "restore_requested", name });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// FILES
app.get("/api/files", requireApiKey, async (req, res) => {
  const api = mk();
  try {
    await safeConnect(api);
    const files = await api.write("/file/print");
    api.close();
    res.json(files);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// HOTSPOT FULL SNAPSHOT
app.get("/api/hotspot/full", requireApiKey, async (req, res) => {
  const api = mk();
  try {
    await safeConnect(api);
    const results = await Promise.all([
      api.write("/ip/hotspot/user/print"),
      api.write("/ip/hotspot/active/print"),
      api.write("/ip/hotspot/user/profile/print"),
      api.write("/ip/hotspot/profile/print"),
      api.write("/ip/hotspot/ip-binding/print"),
      api.write("/ip/hotspot/host/print")
    ]);
    api.close();
    res.json({
      updated: new Date().toISOString(),
      users: results[0],
      active: results[1],
      user_profiles: results[2],
      server_profiles: results[3],
      ip_bindings: results[4],
      hosts: results[5]
    });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// HOTSPOT USERS CRUD
app.get("/api/hotspot/user", requireApiKey, async (req, res) => {
  const api = mk();
  try {
    await safeConnect(api);
    const list = await api.write("/ip/hotspot/user/print");
    api.close();
    res.json(list);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.post("/api/hotspot/user", requireApiKey, async (req, res) => {
  const { name, password, profile = "default", comment = "", disabled = "no" } = req.body;
  if (!name || !password) return res.status(400).json({ error: "name and password required" });
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/ip/hotspot/user/add", [
      `=name=${name}`,
      `=password=${password}`,
      `=profile=${profile}`,
      `=comment=${comment}`,
      `=disabled=${disabled}`
    ]);
    api.close();
    res.json({ status: "user_created", name });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.put("/api/hotspot/user/:id", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const { name, password, profile, comment, disabled } = req.body;
  const api = mk();
  try {
    await safeConnect(api);
    const args = [`=.id=${id}`];
    if (name) args.push(`=name=${name}`);
    if (password) args.push(`=password=${password}`);
    if (profile) args.push(`=profile=${profile}`);
    if (comment) args.push(`=comment=${comment}`);
    if (disabled !== undefined) args.push(`=disabled=${disabled}`);
    await api.write("/ip/hotspot/user/set", args);
    api.close();
    res.json({ status: "user_updated", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.delete("/api/hotspot/user/:id", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/ip/hotspot/user/remove", [`=.id=${id}`]);
    api.close();
    res.json({ status: "user_deleted", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// DISABLE / ENABLE USER
app.post("/api/hotspot/user/:id/disable", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/ip/hotspot/user/set", [`=.id=${id}`, `=disabled=yes`]);
    api.close();
    res.json({ status: "user_disabled", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.post("/api/hotspot/user/:id/enable", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/ip/hotspot/user/set", [`=.id=${id}`, `=disabled=no`]);
    api.close();
    res.json({ status: "user_enabled", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// HOTSPOT ACTIVE SESSION
app.get("/api/hotspot/active", requireApiKey, async (req, res) => {
  const api = mk();
  try {
    await safeConnect(api);
    const active = await api.write("/ip/hotspot/active/print");
    api.close();
    res.json(active);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.delete("/api/hotspot/active/:id", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/ip/hotspot/active/remove", [`=.id=${id}`]);
    api.close();
    res.json({ status: "disconnected", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// VOUCHERS (BATCH)
app.post("/api/hotspot/users/batch", requireApiKey, async (req, res) => {
  const { prefix = "V", count = 10, passwordLength = 6, profile = "default" } = req.body;
  const api = mk();
  try {
    await safeConnect(api);
    const created = [];
    for (let i = 0; i < count; i++) {
      const username = `${prefix}${Date.now().toString(36).slice(-5)}${i}`;
      const password = Math.random().toString(36).slice(2, 2 + passwordLength);
      await api.write("/ip/hotspot/user/add", [
        `=name=${username}`,
        `=password=${password}`,
        `=profile=${profile}`
      ]);
      created.push({ username, password });
    }
    api.close();
    res.json({ status: "created", created });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// USER PROFILES CRUD
app.get("/api/hotspot/user-profile", requireApiKey, async (req, res) => {
  const api = mk();
  try {
    await safeConnect(api);
    const list = await api.write("/ip/hotspot/user/profile/print");
    api.close();
    res.json(list);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.post("/api/hotspot/user-profile", requireApiKey, async (req, res) => {
  const { name, rateLimit, sharedUsers } = req.body;
  if (!name) return res.status(400).json({ error: "name required" });
  const api = mk();
  try {
    await safeConnect(api);
    const args = [`=name=${name}`];
    if (rateLimit) args.push(`=rate-limit=${rateLimit}`);
    if (sharedUsers) args.push(`=shared-users=${sharedUsers}`);
    await api.write("/ip/hotspot/user/profile/add", args);
    api.close();
    res.json({ status: "profile_created", name });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.put("/api/hotspot/user-profile/:id", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const { name, rateLimit, sharedUsers } = req.body;
  const api = mk();
  try {
    await safeConnect(api);
    const args = [`=.id=${id}`];
    if (name) args.push(`=name=${name}`);
    if (rateLimit) args.push(`=rate-limit=${rateLimit}`);
    if (sharedUsers) args.push(`=shared-users=${sharedUsers}`);
    await api.write("/ip/hotspot/user/profile/set", args);
    api.close();
    res.json({ status: "profile_updated", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.delete("/api/hotspot/user-profile/:id", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/ip/hotspot/user/profile/remove", [`=.id=${id}`]);
    api.close();
    res.json({ status: "profile_deleted", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// SERVER PROFILES CRUD
app.get("/api/hotspot/server-profile", requireApiKey, async (req, res) => {
  const api = mk();
  try {
    await safeConnect(api);
    const list = await api.write("/ip/hotspot/profile/print");
    api.close();
    res.json(list);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.post("/api/hotspot/server-profile", requireApiKey, async (req, res) => {
  const { name, dnsName, htmlDirectory, rateLimit } = req.body;
  if (!name) return res.status(400).json({ error: "name required" });
  const api = mk();
  try {
    await safeConnect(api);
    const args = [`=name=${name}`];
    if (dnsName) args.push(`=dns-name=${dnsName}`);
    if (htmlDirectory) args.push(`=html-directory=${htmlDirectory}`);
    if (rateLimit) args.push(`=rate-limit=${rateLimit}`);
    await api.write("/ip/hotspot/profile/add", args);
    api.close();
    res.json({ status: "server_profile_created", name });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.put("/api/hotspot/server-profile/:id", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const { name, dnsName, htmlDirectory, rateLimit } = req.body;
  const api = mk();
  try {
    await safeConnect(api);
    const args = [`=.id=${id}`];
    if (name) args.push(`=name=${name}`);
    if (dnsName) args.push(`=dns-name=${dnsName}`);
    if (htmlDirectory) args.push(`=html-directory=${htmlDirectory}`);
    if (rateLimit) args.push(`=rate-limit=${rateLimit}`);
    await api.write("/ip/hotspot/profile/set", args);
    api.close();
    res.json({ status: "server_profile_updated", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.delete("/api/hotspot/server-profile/:id", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/ip/hotspot/profile/remove", [`=.id=${id}`]);
    api.close();
    res.json({ status: "server_profile_deleted", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// IP BINDING CRUD
app.get("/api/hotspot/ip-binding", requireApiKey, async (req, res) => {
  const api = mk();
  try {
    await safeConnect(api);
    const list = await api.write("/ip/hotspot/ip-binding/print");
    api.close();
    res.json(list);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.post("/api/hotspot/ip-binding", requireApiKey, async (req, res) => {
  const { mac, address, type = "bypassed", comment = "" } = req.body;
  if (!mac && !address) return res.status(400).json({ error: "mac or address required" });
  const api = mk();
  try {
    await safeConnect(api);
    const args = [];
    if (mac) args.push(`=mac-address=${mac}`);
    if (address) args.push(`=address=${address}`);
    args.push(`=type=${type}`);
    args.push(`=comment=${comment}`);
    await api.write("/ip/hotspot/ip-binding/add", args);
    api.close();
    res.json({ status: "ip_binding_created" });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.put("/api/hotspot/ip-binding/:id", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const { mac, address, type, comment } = req.body;
  const api = mk();
  try {
    await safeConnect(api);
    const args = [`=.id=${id}`];
    if (mac) args.push(`=mac-address=${mac}`);
    if (address) args.push(`=address=${address}`);
    if (type) args.push(`=type=${type}`);
    if (comment) args.push(`=comment=${comment}`);
    await api.write("/ip/hotspot/ip-binding/set", args);
    api.close();
    res.json({ status: "ip_binding_updated", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.delete("/api/hotspot/ip-binding/:id", requireApiKey, async (req, res) => {
  const id = req.params.id;
  const api = mk();
  try {
    await safeConnect(api);
    await api.write("/ip/hotspot/ip-binding/remove", [`=.id=${id}`]);
    api.close();
    res.json({ status: "ip_binding_deleted", id });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// LOGS
app.get("/api/hotspot/logs", requireApiKey, async (req, res) => {
  const { topics } = req.query;
  const api = mk();
  try {
    await safeConnect(api);
    const args = topics ? [`?topics=${topics}`] : [];
    const data = await api.write("/log/print", args);
    api.close();
    res.json(data);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// SEARCH USER
app.get("/api/hotspot/user/search", requireApiKey, async (req, res) => {
  const { name, mac } = req.query;
  const api = mk();
  try {
    await safeConnect(api);
    let args = [];
    if (name) args.push(`?name=${name}`);
    if (mac) args.push(`?mac-address=${mac}`);
    if (args.length === 0) return res.status(400).json({ error: "name or mac required" });
    const result = await api.write("/ip/hotspot/user/print", args);
    api.close();
    res.json(result);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// STATS
app.get("/api/hotspot/stats", requireApiKey, async (req, res) => {
  const api = mk();
  try {
    await safeConnect(api);
    const [users, active, ipBindings] = await Promise.all([
      api.write("/ip/hotspot/user/print"),
      api.write("/ip/hotspot/active/print"),
      api.write("/ip/hotspot/ip-binding/print")
    ]);
    api.close();

    const totalUsers = Array.isArray(users) ? users.length : 0;
    const activeUsers = Array.isArray(active) ? active.length : 0;
    const blocked = Array.isArray(users)
      ? users.filter(u => u.disabled === "yes").length
      : 0;

    res.json({
      updated: new Date().toISOString(),
      totalUsers,
      activeUsers,
      blocked,
      bindings: ipBindings.length
    });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

app.listen(PORT, () => console.log("API ON PORT", PORT));